/*
** SIMPLE DISK PERFORMANCE BENCHMARK TEST
** written in Microsoft C v5
**
** uses only legal DOS calls.
** for best results, run on an empty disk.
*/

char banner[] = "MS-DOS Disk Benchmark Version 1.7 by John Navas II\n";

#include <conio.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include <process.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/timeb.h>

#define FALSE	0
#define TRUE	(!FALSE)

#define PRE		20				/* larger makes slower but more precise */

typedef char BOOL;
typedef unsigned long LONG;

char nam[] = "\\dbXXXXXX";			/* temporary file name (at root) */
int fil = EOF;						/* temporary file */

BOOL stdoutty = FALSE;				/* true if stdout is a tty */

int cleanup(void)					/* CLEAN UP ON EXIT */
{
	if ( *nam != '\0' ) remove( nam );	/* delete temporary file */
	*nam = '\0';						/* flag deleted */
	return 0;
	}

int abandon(void)					/* USER WANTS TO ABANDON */
{
	if ( fil != EOF ) close( fil );		/* attempt to close file */
	fil = EOF;							/* flag closed */
	cleanup();							/* delete temp file */
	abort();							/* and kill the program */
	return 0;							/* (dummy) */
	}

int printr(char *str, ...)			/* STANDARD OUTPUT WITH TTY RETURN */
{
	va_list args;

	va_start( args, str );
	if ( stdoutty ) fputchar( '\r' );		/* return if a tty */
	return vprintf( str, args );			/* use vector printf function */
	va_end( args );
	}

char *fmtlong(char *ret, LONG l)	/* FORMAT A LONG TO PRINT 1 DECIMAL */
{
	sprintf( ret, "%4u.%01u", (unsigned) ( l / 10 ), (unsigned) ( l % 10 ) );
	return ret;
	}

LONG millisec(						/* COMPUTE TIME DIFFERENCE IN MILLISEC */
struct timeb *start,
struct timeb *stop)
{
	return 1000L * ( stop->time - start->time ) +
		stop->millitm - start->millitm;
	}

char *dma_buffer(unsigned int siz)	/* GET A LEGAL BIOS BUFFER */
{
	char *buf, far *lo;
	unsigned long adr, len;

	if ( ( buf = malloc( siz ) ) == NULL ) {	/* allocate */
		fprintf( stderr, "Buffer Allocation Failure\n" );
		abort();
		}
	lo = buf;									/* check for 64K boundary */
	adr = ( (unsigned long) FP_SEG( lo ) << 4 ) + FP_OFF( lo );
	if ( ( len = ( ( adr + 0x10000L ) & ~0xFFFFL ) - adr ) < siz ) {
		buf = realloc( buf, (unsigned int) len );	/* put aside the hole */
		lo = dma_buffer( siz );						/* try again */
		free( buf );								/* release hole */
		}
	return (char *) lo;
	}

LONG readwrite(						/* TIME READ OR WRITE */
enum FILOP { NUL, READ, WRITE } op,		/* file operation */
unsigned num,							/* number of I/O */
unsigned len)							/* length of I/O */
{
	char *buf;							/* i/o buffer */
	struct timeb start, stop;			/* times */

	if ( op != NUL ) buf = dma_buffer( len );	/* allocate buffer */
	kbhit();							/* check for break */
	lseek( fil, 0L, SEEK_SET );			/* beginning of file */
	ftime( &start );					/* starting time */
	switch ( op ) {
		case READ:	while ( num-- )
						if ( read( fil, buf, len ) != len ) {
							perror( "READING" );
							abort();
							}
					break;
		case WRITE:	while ( num-- )
						if ( write( fil, buf, len ) != len ) {
							perror( "WRITING" );
							abort();
							}
					break;
		}
	ftime( &stop );						/* ending time */
	if ( op != NUL ) free( buf );		/* free i/o buffer */
	return millisec( &start, &stop );	/* elapsed time */
	}

void timetype(						/* TIME TYPE OF FILE I/O */
unsigned num,							/* number of I/O */
unsigned len,							/* length of I/O */
char *lbl)								/* type label */
{
	int r = 0, w = 0;					/* read/write passes */
	LONG nul, rtim = 0L, wtim = 0L;		/* timer values */
	char rb[ 7 ], wb[ 7 ];				/* character formatting */

	fprintf( stderr, "\rTiming  %s I/O...", lbl );
	nul = readwrite( NUL, 0, 0 );			/* timer overhead */
	do wtim += readwrite( WRITE, num, len ) - nul;
		while ( ++w, wtim < 1000L * PRE );		/* write file time */
	do rtim += readwrite( READ, num, len ) - nul;
		while ( ++r, rtim < 1000L * PRE );		/* read file time */
	printr( "%s I/O    read %s kb/sec.    write %s kb/sec.\n", lbl,
		fmtlong( rb, ( ( (LONG) len * (LONG) num * (LONG) r * 100L ) / rtim
			+ 5 ) / 10 ),
		fmtlong( wb, ( ( (LONG) len * (LONG) num * (LONG) w * 100L ) / wtim
			+ 5 ) / 10 ) );
	}

void timefileio(void)				/* TIME FILE I/O OPERATIONS */
{
	fil = open( mktemp( nam ),
		(int) ( O_RDWR+O_CREAT+O_TRUNC+O_BINARY ),
		(int) ( S_IREAD+S_IWRITE ) );
	if ( fil < 0 ) {
		perror( "OPENING TEMP FILE" );
		abort();
		}
	fprintf( stderr, "\rCreating temp file..." );
	readwrite( WRITE, 10, 32768 );	/* create file */
	timetype( 11, (unsigned) 32768, "Block " );	/* time block I/O */
	timetype( 110, 512, "Sector" );				/* time sector I/O */
	if ( close( fil ) ) {				/* close temporary file */
		perror( "CLOSING TEMP FILE" );
		abort();
		}
	fil = EOF;							/* flagged closed */
	cleanup();							/* delete temp file */
	}

int secread(						/* RANDOM SECTOR READ */
unsigned char drv,						/* drive number */
char *buf,								/* i/o buffer */
unsigned cnt,							/* number of sectors */
unsigned sec)							/* starting sector number */
{
	union REGS reg;

	reg.h.al = drv;						/* drive number */
	reg.x.bx = (unsigned int) buf;		/* i/o buffer */
	reg.x.cx = cnt;						/* one sector */
	reg.x.dx = sec;						/* starting sector */
	int86( 0x25, &reg, &reg );			/* absolute DOS sector read */
	return reg.x.cflag;
	}

void timeseekread(void)				/* TIME DOS SEEK-READ 1 SECTOR */
{
	struct timeb start, stop;			/* times */
	register unsigned int i, cnt;		/* number of i/o's */
	unsigned char drv =					/* drive number */
		(unsigned char) bdos( 0x19, 0, 0 );	/* get default drive */
	unsigned sec;						/* total sectors */
	char *buf, bf[ 7 ];					/* i/o & display buffers */

	{	union REGS reg;
		struct SREGS sreg;
	
		reg.h.ah = 0x1B;				/* get alloc info for default drv */
		intdosx( &reg, &reg, &sreg );
		sec = reg.x.dx * reg.h.al;		/* sectors = clusters * secperclus */
		buf = dma_buffer( reg.x.cx );	/* i/o buffer */
		}

	fprintf( stderr, "\rTiming DOS random sector read..." );
RESTART:
	secread( drv, buf, 1, sec / 2 );	/* prepare the drive */
	cnt = 0; ftime( &start );			/* starting time */
	do for ( i = 0; i < 25; ++i )		/* 25 random seek-reads */
	   	if ( secread( drv, buf, 1,
				(unsigned int ) ( ( ( (LONG) rand() << 16 ) |
				(unsigned) rand() ) % sec ) ) ) {	/* truly random sector */
			printr( "I/O error, restarting timing... " );
			goto RESTART;
			}
		while( ftime( &stop ), cnt += i, stop.time - start.time < PRE );
	printr( "DOS random sector read %s ms. avg.\n",
		fmtlong( bf, ( millisec( &start, &stop ) * 100L / cnt + 5 ) / 10 ) );

	fprintf( stderr, "\rTiming rotation speed of drive..." );
	secread( drv, buf, 1, 0 );			/* sync with drive sector 0 */
	cnt = 0; ftime( &start );			/* starting time */
	do for ( i = 0; i < 250; ++i )
			secread( drv, buf, 1, 0 );	/* repeatedly read same sector */
		while( ftime( &stop ), cnt += i, stop.time - start.time < PRE );
	printr( "Effective rotation speed  %5lu rpm\n",
		60000L * cnt / millisec( &start, &stop ) );

	free( buf );						/* free i/o buffer */
	}

main()
{
	fprintf( stderr, banner );
	if ( isatty( fileno( stdout ) ) ) stdoutty = TRUE;
	onexit( cleanup );					/* cleanup routine */
	signal( SIGINT, abandon );			/* if user wants to abandon */
	timefileio();						/* time DOS file I/O */
	timeseekread();						/* time seek-read I/O */
	fprintf( stderr, "\r%-40s\n", "Processing completed" );
	return 0;
	}
