#include "stdio.h"

#define fabs(A) (((A)>0.0) ? (A) : -(A))
#define sign(x) ((x == 0.0) ? 1.0 : x/fabs(x))

#define MAXDIM 81
/* MAXDIM = 81 is a little dangerous. If you are storing much else use 41.*/

/* qsol.c -- Solve linear systems by QR factorization

   Functions in this file:
		qsol()
		qrdecm()
		qrsolv()
		rsolv()

   m is an array of pointers to rows of matrix, b is right hand side,
   x is solution, n is dimension of the problem. Indices vary from 0 to
   n-1, calling program must adjust to this.

   The algorithm is the one given in G.W. Stewart's book, Introduction to
   Matrix Computations. The code is based on the algorithms given in
   Dennis and Schnabel's book, Numerical Methods for Unconstrained Optimization
   and Nonlinear Equations. If you are unfamiliar with these ideas you should
   consult one or both of these books.

*/

qsol(m,b1,x,n)

double *m[],b1[],x[];
int n;

/* This is the driver function for the QR decomposition routines and
   the solvers. In using these you should be sure to pass an array of
   pointers to the rows of the matrix rather than the matrix itself.
   That is the setup should include something like

	for(i=0;i<n;i++) m[i]=a[i];

   where a is the matrix and m is an array of pointers allocated for this
   purpose. If you don't do this many compilers will be upset.

   n is the dimension of the problem.
   b1 is the right hand side.
   The solution is returned in x.
   MAXDIM should be defined in an include file. The value 41 is safe for
   small model code. Note that the assembler routines assume small model
   and would have to be reworked to use the large model.
*/

{
	double m1[MAXDIM],m2[MAXDIM],c[MAXDIM];
	int i,*isng,sgt;

	/* The vector b1 is copied to c because qrsolv overwrites the right
	   hand side with the solution.
	*/

	for(i=0;i<n;i++) c[i]=b1[i];

	isng= &sgt;

	qrdecm(m,m1,m2,n,isng);

	if((*isng) == 1) {
		printf("singularity detected in QR");
		exit(1);
	}

	qrsolv(m,m1,m2,c,n);

	for(i=0;i<n;i++) x[i]=c[i];
}

qrdecm(m,m1,m2,n,isng)
double *m[],m1[],m2[];
int n,*isng;

/* Form the QR decomposition of an nxn matrix. m is an array of pointers to
   the rows of the matrix. m1 and m2 are work arrays. The decomposition is
   stored in m, m1, and m2 as in Stewart's book. isng is a pointer to an
   int and is set to 1 if the matrix is singular. This should be tested on
   return to avoid a possible zero divide in qsolv.
*/

{
	double eta,sigma,tau,sum,*work1,*work2,mtmp;
	int i,k,j,stride;

	/* Stride is the distance in bytes between consecutive rows. Note
	   the implicit assumption of small model code here. When calling
	   those assembler routines that need stride be sure to multiply
	   the pointer difference by 8. I know that I am using the fact
	   that in the small model sizeof(int)=sizeof(*int) but I don't
	   care.
	*/
	stride=m[1]-m[0];
	stride*=8;

	*isng=0;

	for(k=0;k<n-1;k++)
	{
	       work1=&m[k][k];
	       dnrm2(work1,n-k,&sum,stride);

	       /* Test for singularity. */

	       if(sum < 1.0e-15) {
			*isng=1;
			m1[k]=0.0;
			m2[k]=0.0;
		}

		else
		{
			sigma=sign(m[k][k])*sum;
			m[k][k]+=sigma;
			m1[k]=sigma*m[k][k];
			mtmp=m1[k];
			m2[k]= -sigma;

			/* This is the critical loop. */
			for(j=k+1;j<n;j++)
			{
				work2=&m[k][j];
				ssxdot(work1,work2,n-k,&tau,stride,stride);
				ssdaxpy(work1,work2,n-k,-tau/mtmp,stride,stride);
			}

		}
	}
	m2[n-1]=m[n-1][n-1];
}


qrsolv(m,m1,m2,b,n)
double *m[],m1[],m2[],b[];
int n;

/* The qrsolv() funciton uses the results of qrdecm to solve Ax=b. The vector
   b is over written by the solution x.
*/

{
	double tau;
	int i,j,stride;
	/* Compute the stride. */
	stride=(m[1]-m[0])*8;

	/* Multiply b by the transpose of Q. */

	for(j=0;j<n-1;j++)
	{
		ssxdot(&m[j][j],&b[j],n-j,&tau,stride,8);
		ssdaxpy(&m[j][j],&b[j],n-j,-tau/m1[j],stride,8);
	}

	/* Solve the resulting triangular system. */

	rsolv(n,m,m2,b);

}

/* Solve a triangular system. */

rsolv(n,m,m2,b)
double *m[],m2[],b[];
int n;

{
	double sum;
	int i,j,n1;
	n1=n-1;
	b[n1]/=m2[n1];

	/* We assume that the vector b is stored in consecutive locations
	   in memory so stride is sizeof(double)=8.
	*/

	for(i=n1-1;i>=0;i--)
	{
		ssxdot(m[i]+i+1,b+i+1,n-i-1,&sum,8,8);
		b[i]-=sum;
		b[i]/=m2[i];
	}
}
